
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Mon Mar 28 12:09:38 2005
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief A brief description of this module
   *
   *  \version SST_TST_ATP_Authenticators.c#1:csrc:1
   *  \author doronp
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ***********************************/
#include "SST_TST.h"
#if TST_USE_AUTHENTICATOR_MODULE
/************************ Defines ******************************/
/************************ Enums ********************************/
/************************ Typedefs *****************************/
/************************ Global Data **************************/
/************* Private function prototype **********************/
/************************ Public Functions *********************/
/*****************************************************************************
 * Function Name:                                                           
 *  SST_TST_ATP_PasswordAuthenticator         
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  DxError_t - Function error return                                        
 *
 * Description:                                                                 
 * Test the creation, deletion, modification, opening, 
 * closing and usage of a password authenticator object. 
 * And demonstrate the use of transaction with authenticator objects
 *  
 * Algorithm:       
 *    1.  Create a new session by calling: SST_SessionCreate.
 *    2.  Open a transaction by calling: SST_TransactionStart.
 *    3.  Create a new master password authenticator by calling: SST_AuthenticatorPasswordCreate.
 *    4.  Close the transaction by calling: SST_TransactionEnd
 *    5.  Open the master password authentication by calling: SST_AuthenticationOpen.
 *    6.  Open a transaction by calling: SST_TransactionStart.
 *    7.  Create a new secondary password authenticator by calling: SST_AuthenticatorPasswordCreate.
 *    8.  Close the transaction by calling: SST_TransactionEnd. 
 *    9.  Open the secondary authenticator by calling: SST_AuthenticationOpen.
 *    10.  Open a transaction by calling: SST_TransactionStart.
 *    11.  Modify the password for the authenticator by calling: SST_AuthenticatorPasswordResetAndChange.
 *    12.  Close the authenticator by calling: SST_AuthenticationClose.
 *    13.  Close the transaction by calling: SST_TransactionEnd.
 *    14.  Reopen the secondary authenticator using the new password by calling: SST_AuthenticationOpen.
 *    15.  Close the authenticator by calling: SST_AuthenticationClose.
 *    16.  Open the master authenticator using the new password by calling: SST_AuthenticationOpen.
 *    17.  Open a transaction by calling: SST_TransactionStart.
 *    18.  Insert a new data object by calling: SST_InsertData.
 *    19.  Delete the data by calling: SST_DeleteData.
 *    20.  Close the authenticator by calling: SST_AuthenticationClose.
 *    21.  Delete the authenticators (one by one) by calling: SST_AuthenticatorDelete.
 *    22.  Close the transaction by calling: SST_TransactionClose.
 *    23.  Delete the session by calling: SST_SessionDelete.
 *                                
 *******************************************************************************/    
DxError_t SST_TST_ATP_PasswordAuthenticator ( void )
{
    /********* TST Variables Definitions ************/
    int                         TST_index,TST_HandleType;
    char                        TST_MasterTestName[TST_FUNC_STRING_LEN] = "SST_TST_ATP_PasswordAuthenticator";
    SSTSessionId_t              TST_SessionId; 
    SSTTxnId_t                  TST_TransactionId;
    SSTHandle_t                 TST_MasterPasswordAuthHandle,TST_SecondaryPasswordAuthHandle,TST_DataHandle;
    SSTDataType_t               TST_SimpleData = TST_DATA_OBJECT;
    SSTHandle_t                 TST_AuthDefultHandle;
    DxError_t                   TST_Status;
    /********* TST Variables initialization *********/

    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************************** \n"));
    ATP_LOG_TST_PRINT((SST_TST_MSG," ** SST_TST_ATP_PasswordAuthenticator Tests ********************* \n"));     
    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************************** \n \n"));

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    TST_Status = TESTPASS;
    for (TST_HandleType = TST_INTERNAL_HANDLE ; TST_HandleType < TST_HANDLES_TYPES ; TST_HandleType++)
    {
        for (TST_index =0 ; TST_index < TST_NumberOf_PassAuthObjectTests ; TST_index++) /*To run on all test vectors*/
        {
            if (TST_HandleType == TST_INTERNAL_HANDLE)
            {   
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using internal handles          \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                TST_MasterPasswordAuthHandle.objId = SST_HANDLE_RANDOM_GENERATE; 
                TST_SecondaryPasswordAuthHandle.objId = SST_HANDLE_RANDOM_GENERATE;
                TST_DataHandle.objId = SST_HANDLE_RANDOM_GENERATE;
            }
            else
            {
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using external handles          \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                TST_MasterPasswordAuthHandle.objId = TST_TEST_DEFINED_HENDLE;
                TST_SecondaryPasswordAuthHandle.objId = TST_TEST_DEFINED_HENDLE_1;
                TST_DataHandle.objId = TST_TEST_DEFINED_HENDLE;
            }  

            ATP_LOG_TST_PRINT((SST_TST_MSG,"\n",TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,"\n"));

            /* Create a session for operations */
            /*---------------------------------*/
            TST_Status = SST_SessionCreate(&TST_SessionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionCreate",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Create a new master password authenticator */
            /*--------------------------------------------*/
             TST_AuthDefultHandle.objDigest = 0;
       TST_AuthDefultHandle.objId = SST_CLOSE_AUTH_HANDLE;
             TST_Status = SST_AuthenticatorPasswordCreate(TST_TransactionId,
                                                        TST_SessionId, 
                                                        TST_ATP_PassAuthObjectTestData[TST_index].TST_MasterPassword,
                                                        TST_ATP_PassAuthObjectTestData[TST_index].TST_MasterPasswordSizeInBytes,
                                                        TST_ATP_PassAuthObjectTestData[TST_index].TST_MasterMaxNumOfTrials,
                                                        TST_AuthDefultHandle,
                                                        &TST_MasterPasswordAuthHandle,
                                                        TST_Workspace,
                                                        TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorPasswordCreate - Master Authenticator",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the master password authentication */
            /*-----------------------------------------*/
            TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                               TST_MasterPasswordAuthHandle,
                                               TST_ATP_PassAuthObjectTestData[TST_index].TST_MasterPassword,
                                               TST_ATP_PassAuthObjectTestData[TST_index].TST_MasterPasswordSizeInBytes,
                                               TST_NO_CHALLENGE,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);               
                                               

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen - Master Authentication",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);



            /* Create a new password authenticator */
            /*-------------------------------------*/
            TST_Status = SST_AuthenticatorPasswordCreate(TST_TransactionId,
                                                        TST_SessionId,
                                                        TST_ATP_PassAuthObjectTestData[TST_index].TST_SecondaryPassword,
                                                        TST_ATP_PassAuthObjectTestData[TST_index].TST_SecondaryPasswordSizeInBytes,
                                                        TST_ATP_PassAuthObjectTestData[TST_index].TST_SecondaryMaxNumOfTrials,
                                                        TST_MasterPasswordAuthHandle,
                                                        &TST_SecondaryPasswordAuthHandle,
                                                        TST_Workspace,
                                                        TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorPasswordCreate - Secondary Authenticator",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the secondary authentication */
            /*-----------------------------------*/
            TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                               TST_SecondaryPasswordAuthHandle,
                                               TST_ATP_PassAuthObjectTestData[TST_index].TST_SecondaryPassword,
                                               TST_ATP_PassAuthObjectTestData[TST_index].TST_SecondaryPasswordSizeInBytes,
                                               TST_NO_CHALLENGE,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);         
                                               

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen - Secondary Authentication",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);

            
      /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Modify the secondary authenticator */
            /*------------------------------------*/
            TST_Status = SST_AuthenticatorPasswordResetAndChange(TST_TransactionId,
                                                                TST_SessionId,
                                TST_SecondaryPasswordAuthHandle,
                                                                TST_ATP_PassAuthObjectTestData[TST_index].TST_NewSecPasswordSizeInBytes,                    
                                                                TST_ATP_PassAuthObjectTestData[TST_index].TST_NewSecPassword,
                                                                TST_ATP_PassAuthObjectTestData[TST_index].TST_NewSecMaxNumOfTrials,
                                                                TST_Workspace,
                                                                TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorPasswordResetAndChange",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the secondary authentication */
            /*------------------------------------*/
            TST_Status = SST_AuthenticationClose(TST_SessionId,TST_SecondaryPasswordAuthHandle);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose ",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* Close the transaction */
            /*-----------------------*/
      TST_Status = SST_TransactionEnd(TST_TransactionId,
                      DX_TRUE,/*reflash sensitive*/
                      TST_Workspace,
                      TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);

            /* Reopen the secondary authenticator using the new password */
            /*-----------------------------------------------------------*/
            TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                               TST_SecondaryPasswordAuthHandle,
                                               TST_ATP_PassAuthObjectTestData[TST_index].TST_NewSecPassword,
                                               TST_ATP_PassAuthObjectTestData[TST_index].TST_NewSecPasswordSizeInBytes,
                                               TST_NO_CHALLENGE,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);               
                                               

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen - Secondary Authentication new password",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the secondary authentication */
            /*------------------------------------*/
            TST_Status = SST_AuthenticationClose(TST_SessionId,TST_SecondaryPasswordAuthHandle);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose SecondaryPasswordAuthHandle",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);



            /* Open the Master authentication */
            /*--------------------------------*/
            TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                               TST_MasterPasswordAuthHandle,
                                               TST_ATP_PassAuthObjectTestData[TST_index].TST_MasterPassword,
                                               TST_ATP_PassAuthObjectTestData[TST_index].TST_MasterPasswordSizeInBytes,
                                               TST_NO_CHALLENGE,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);         
                                               

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen - Secondary Authentication",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);

      
      /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Insert a new Data object */
            /*--------------------------*/
            TST_Status = SST_DataInsert(TST_TransactionId,
                                       TST_SessionId, 
                                       TST_MasterPasswordAuthHandle,       
                                       TST_ATP_PassAuthObjectTestData[TST_index].TST_Data,
                                       TST_ATP_PassAuthObjectTestData[TST_index].TST_DataSize,
                                       DX_TRUE,
                                       TST_SimpleData,                
                                       &TST_DataHandle,
                                       TST_Workspace,
                                       TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataInsert",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the data */
            /*-----------------*/
             TST_Status = SST_DataDelete(TST_TransactionId,
                                        TST_SessionId,  
                                        TST_DataHandle,
                                        TST_Workspace,
                                        TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataDelete",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the master authentication */
            /*---------------------------------*/
            TST_Status = SST_AuthenticationClose(TST_SessionId,TST_MasterPasswordAuthHandle);
            
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the authenticators */
            /*---------------------------*/
            TST_Status = SST_AuthenticatorDelete(TST_TransactionId,
                                                TST_SecondaryPasswordAuthHandle);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorDelete - Secondary authenticator",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            TST_Status = SST_AuthenticatorDelete(TST_TransactionId,
                                                TST_MasterPasswordAuthHandle);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorDelete - Master authenticator",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);



            /* Close the transaction */
            /*-----------------------*/
        TST_Status = SST_TransactionEnd(TST_TransactionId,
                        DX_TRUE,/*reflash sensitive*/
                        TST_Workspace,
                        TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the session */
            /*--------------------*/
            TST_Status = SST_SessionDelete(TST_SessionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionDelete",
                          TST_ATP_PassAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


        }/*End of TST_index "for loop" */
    }/*End of TST_HandleType "for loop" */
    return TST_Status;

  EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}/* End of SST_TST_ATP_PasswordAuthenticator */

/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*****************************************************************************
 * Function Name:                                                           
 *  SST_TST_ATP_RsaAsymAuthenticator         
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  DxError_t - Function error return                                        
 *
 * Description:                                                                 
 * Test the creation, deletion, modification, opening, closing and usage of an asymmetric key authenticator object.
 * And demonstration on the use of transaction with authenticator objects.
 *  
 * Algorithm:       
 *    1.  Build a Private Key using CRYS for CRYS usage by calling: CRYS_RSA_Build_PrivKey (externally generated key).
 *    2.  Create a new session by calling: SST_SessionCreate.
 *    3.  Open a transaction by calling: SST_TransactionStart.
 *    4.  Create a new asymmetric key authenticator by calling: SST_AuthenticatorPubKeyRSACreate.
 *    5.  Close the transaction by calling: SST_TransactionClose.
 *    6.  Open a transaction by calling: SST_TransactionStart.
 *    7.  Open the new authenticator by calling: 
 *        i.    SST_AuthenticatorChallengeGet 
 *        ii.  CRYS_RSA_PSS_Sign
 *        iii.  SST_AuthenticationOpen.
 *    8.  Close the transaction by calling: SST_TransactionClose.
 *    9.  Open a transaction by calling: SST_TransactionStart.
 *    10.  Insert a new data object by calling: SST_InsertData.
 *    11.  Close the authenticator by calling: SST_AuthenticationClose.
 *    12.  Open the authenticator by calling: 
 *        i.      SST_AuthenticatorChallengeGet 
 *        ii.  CRYS_RSA_PSS_Sign
 *        iii.  SST_AuthenticationOpen.
 *    13.  Close the transaction by calling: SST_TransactionClose.
 *    14.  Open a transaction by calling: SST_TransactionStart.
 *    15.  Delete the data by calling: SST_DeleteData.
 *    16.  Delete the authenticator by calling: SST_AuthenticatorDelete.
 *    17.  Close the transaction by calling: SST_TransactionClose.
 *    18.  Delete the session by calling: SST_SessionDelete.                           
 *
 *******************************************************************************/    
DxError_t SST_TST_ATP_RsaAsymAuthenticator ( void )
{
    /********* TST Variables Definitions ************/
    int                         TST_index,TST_HandleType;
    char                        TST_MasterTestName[TST_FUNC_STRING_LEN] = "SST_TST_ATP_RsaAsymAuthenticator";
    SSTSessionId_t              TST_SessionId; 
    SSTTxnId_t                  TST_TransactionId;
    SSTHandle_t                 TST_AuthHandle,TST_DataHandle;
    SSTChallengeId_t            TST_ChallengeId;
    SSTDataType_t               TST_SimpleData = TST_DATA_OBJECT;
    static CRYS_RSAUserPrivKey_t       TST_CRYS_RSAUserPrivKey;
    static CRYS_RSAPrivUserContext_t   TST_CRYS_RSAPrivUserContext;
    DxError_t                   TST_Status;
    /********* TST Variables initialization *********/

    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************************** \n"));
    ATP_LOG_TST_PRINT((SST_TST_MSG," ** SST_TST_ATP_RsaAsymAuthenticator Tests ********************* \n"));     
    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************************** \n \n"));

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    TST_Status = TESTPASS;
    for (TST_HandleType = TST_INTERNAL_HANDLE ; TST_HandleType < TST_HANDLES_TYPES ; TST_HandleType++)
    {
        for (TST_index =0 ; TST_index < TST_NumberOf_RsaAuthObjectTests ; TST_index++) /*To run on all test vectors*/
        {
            if (TST_HandleType == TST_INTERNAL_HANDLE)
            {   
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using internal handles          \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                TST_AuthHandle.objId = SST_HANDLE_RANDOM_GENERATE; 
                TST_DataHandle.objId = SST_HANDLE_RANDOM_GENERATE;
            }
            else
            {
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using external handles          \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                TST_AuthHandle.objId = TST_TEST_DEFINED_HENDLE;
                TST_DataHandle.objId = TST_TEST_DEFINED_HENDLE;
            }

            ATP_LOG_TST_PRINT((SST_TST_MSG,"\n",TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,"\n"));

            /* Build the Private Key for CRYS usage */
            /*--------------------------------------*/
            TST_Status = CRYS_RSA_Build_PrivKey(&TST_CRYS_RSAUserPrivKey,
                                               TST_ATP_RsaAuthObjectTestData[TST_index].TST_D,
                                               (DxUint16_t)TST_ATP_RsaAuthObjectTestData[TST_index].TST_DSizeInBytes,
                                               TST_ATP_RsaAuthObjectTestData[TST_index].TST_E,
                                               (DxUint16_t)TST_ATP_RsaAuthObjectTestData[TST_index].TST_ESizeInBytes,
                                               TST_ATP_RsaAuthObjectTestData[TST_index].TST_N,
                                               (DxUint16_t)TST_ATP_RsaAuthObjectTestData[TST_index].TST_NSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKey",
                    TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                    TST_MasterTestName);


            /* Create a session for operations */
            /*---------------------------------*/
            TST_Status = SST_SessionCreate(&TST_SessionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"TST_SessionId",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Create a new RSA authenticator */
            /*--------------------------------*/
            TST_Status = SST_AuthenticatorPubKeyRSACreate(TST_TransactionId,
                                                      TST_ATP_RsaAuthObjectTestData[TST_index].TST_N,
                                                      TST_ATP_RsaAuthObjectTestData[TST_index].TST_E,
                                                      TST_ATP_RsaAuthObjectTestData[TST_index].TST_ESizeInBytes,
                                                      &TST_AuthHandle,
                                                      TST_Workspace,
                                                      TST_WorkspaceSizeInBytes);

            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorPubKeyRSACreate",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* re-open a transaction for operations */
            /*--------------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the authentication */
            /*-------------------------*/
            /* Get the challenge*/
            TST_Status = SST_AuthenticatorChallengeGet(TST_ATP_RsaAuthObjectTestData[TST_index].TST_ChallengeBuff,
                                                      TST_ATP_RsaAuthObjectTestData[TST_index].TST_ChallengeBuffSize,  
                                                      &TST_ChallengeId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorChallengeGet",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Sign the Challenge buffer */
            TST_Status = CRYS_RSA_PSS_Sign(&TST_CRYS_RSAPrivUserContext,
                                               &TST_CRYS_RSAUserPrivKey,
                                               CRYS_RSA_HASH_SHA1_mode,
                         CRYS_PKCS1_MGF1,
                         20,
                                               TST_ATP_RsaAuthObjectTestData[TST_index].TST_ChallengeBuff,
                                               TST_ATP_RsaAuthObjectTestData[TST_index].TST_ChallengeBuffSize,
                                               TST_ATP_RsaAuthObjectTestData[TST_index].TST_SignedChallengeBuff,
                                               &TST_ATP_RsaAuthObjectTestData[TST_index].TST_SignedChallengeBuffSize);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PSS_Sign",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the authenticator */
            TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                               TST_AuthHandle,
                                               TST_ATP_RsaAuthObjectTestData[TST_index].TST_SignedChallengeBuff,
                                               TST_ATP_RsaAuthObjectTestData[TST_index].TST_SignedChallengeBuffSize,
                                               TST_ChallengeId,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);               
                                               

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* re-open a transaction for operations */
            /*--------------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Insert a new Data object */
            /*--------------------------*/
            TST_Status = SST_DataInsert(TST_TransactionId,
                                       TST_SessionId, 
                                       TST_AuthHandle,       
                                       TST_ATP_RsaAuthObjectTestData[TST_index].TST_Data,
                                       TST_ATP_RsaAuthObjectTestData[TST_index].TST_DataSize,
                                       DX_TRUE,
                                       TST_SimpleData,                
                                       &TST_DataHandle,
                                       TST_Workspace,
                                       TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataInsert",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);



            /* Close the authentication */
            /*--------------------------*/
            TST_Status = SST_AuthenticationClose(TST_SessionId,TST_AuthHandle);
            
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Re-open the authentication */
            /*----------------------------*/
            /* Get the challenge*/
            TST_Status = SST_AuthenticatorChallengeGet(TST_ATP_RsaAuthObjectTestData[TST_index].TST_NewChallengeBuff,
                                                      TST_ATP_RsaAuthObjectTestData[TST_index].TST_NewChallengeBuffSize,  
                                                      &TST_ChallengeId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorChallengeGet",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Sign the Challenge buffer */
            TST_Status = CRYS_RSA_PSS_Sign(&TST_CRYS_RSAPrivUserContext,
                                               &TST_CRYS_RSAUserPrivKey,
                                               CRYS_RSA_HASH_SHA1_mode,
                         CRYS_PKCS1_MGF1,
                         20,
                                               TST_ATP_RsaAuthObjectTestData[TST_index].TST_NewChallengeBuff,
                                               TST_ATP_RsaAuthObjectTestData[TST_index].TST_NewChallengeBuffSize,
                                               TST_ATP_RsaAuthObjectTestData[TST_index].TST_SignedChallengeBuff,
                                               &TST_ATP_RsaAuthObjectTestData[TST_index].TST_SignedChallengeBuffSize);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PSS_Sign",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);
  

      /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the authenticator */
            TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                               TST_AuthHandle,
                         TST_ATP_RsaAuthObjectTestData[TST_index].TST_SignedChallengeBuff,
                         TST_ATP_RsaAuthObjectTestData[TST_index].TST_SignedChallengeBuffSize,
                                               TST_ChallengeId,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);
                                               

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);
  

      /* re-open a transaction for operations */
            /*--------------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);
    

            /* Delete the data */
            /*-----------------*/
             TST_Status = SST_DataDelete(TST_TransactionId,
                                        TST_SessionId,  
                                        TST_DataHandle,
                                        TST_Workspace,
                                        TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataDelete",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the authentication */
            /*--------------------------*/
            TST_Status = SST_AuthenticationClose(TST_SessionId,TST_AuthHandle);
            
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the authenticator */
            /*--------------------------*/
            TST_Status = SST_AuthenticatorDelete(TST_TransactionId,
                                                TST_AuthHandle);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorDelete",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the session */
            /*--------------------*/
            TST_Status = SST_SessionDelete(TST_SessionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionDelete",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


        }/*End of TST_index "for loop" */
    }/*End of TST_HandleType "for loop" */
    return TST_Status;

  EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}/* End of SST_TST_ATP_RsaAsymAuthenticator */
/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*****************************************************************************
 * Function Name:                                                           
 *  SST_TST_ATP_AesKeyAsymAuthenticator         
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  DxError_t - Function error return                                        
 *
 * Description:                                                                 
 * Test the creation, deletion, modification, opening, closing and usage of a shared key authenticator object.
 * And demonstration on the use of transaction with authenticator objects
 *  
 * Algorithm:       
 *     1.  Create a new session by calling: SST_SessionCreate.
 *     2.  Open a transaction by calling: SST_TransactionStart.
 *     3.  Create a new asymmetric key authenticator by calling: SST_AuthenticatorAESSharedSecretCreate.
 *     4.  Close the transaction by calling: SST_TransactionClose.
 *     5.  Open a transaction by calling: SST_TransactionStart.
 *     6.  Open the new authenticator by calling: 
 *          i.      SST_AuthenticatorChallengeGet 
 *          ii.      CRYS_AES (Encrypt the challenge using the AES key)
 *          iii.  SST_AuthenticationOpen.
 *     7.  Close the transaction by calling: SST_TransactionClose.
 *     8.  Open a transaction by calling: SST_TransactionStart.
 *     9.  Insert a new data object by calling: SST_InsertData.
 *     10.  Close the authenticator by calling: SST_AuthenticationClose.
 *     11.  Open the new authenticator by calling: 
 *          i.      SST_AuthenticatorChallengeGet 
 *          ii.      CRYS_AES (Encrypt the challenge using the AES key)
 *          iii.  SST_AuthenticationOpen.
 *     12.  Close the transaction by calling: SST_TransactionClose.
 *     13.  Open a transaction by calling: SST_TransactionStart.
 *     14.  Delete the data by calling: SST_DeleteData.
 *     15.  Close the authenticator by calling: SST_AuthenticationClose.
 *     16.  Delete the authenticator by calling: SST_AuthenticatorDelete.
 *     17.  Close the transaction by calling: SST_TransactionClose.
 *     18.  Delete the session by calling: SST_SessionDelete.
 * 
 *******************************************************************************/    
DxError_t SST_TST_ATP_AesKeyAsymAuthenticator ( void )
{
    /********* TST Variables Definitions ************/
    int                         TST_index,TST_HandleType;
    char                        TST_MasterTestName[TST_FUNC_STRING_LEN] = "SST_TST_ATP_AesKeyAsymAuthenticator";
    SSTSessionId_t              TST_SessionId; 
    SSTTxnId_t                  TST_TransactionId;
    SSTHandle_t                 TST_AuthHandle,TST_DataHandle;
    SSTChallengeId_t            TST_ChallengeId;
    SSTDataType_t               TST_SimpleData = TST_DATA_OBJECT;
    DxError_t                   TST_Status;
    /********* TST Variables initialization *********/

    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************************** \n"));
    ATP_LOG_TST_PRINT((SST_TST_MSG," ** SST_TST_ATP_AesKeyAsymAuthenticator Tests ********************* \n"));     
    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************************** \n \n"));

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    TST_Status = TESTPASS;
    for (TST_HandleType = TST_INTERNAL_HANDLE ; TST_HandleType < TST_HANDLES_TYPES ; TST_HandleType++)
    {
        for (TST_index =0 ; TST_index < TST_NumberOf_AesKeyAuthObjectTests ; TST_index++) /*To run on all test vectors*/
        {
            if (TST_HandleType == TST_INTERNAL_HANDLE)
            {   
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using internal handles          \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                TST_AuthHandle.objId = SST_HANDLE_RANDOM_GENERATE; 
                TST_DataHandle.objId = SST_HANDLE_RANDOM_GENERATE;
            }
            else
            {
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using external handles          \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                TST_AuthHandle.objId = TST_TEST_DEFINED_HENDLE;
                TST_DataHandle.objId = TST_TEST_DEFINED_HENDLE;
            }

            ATP_LOG_TST_PRINT((SST_TST_MSG,"\n",TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,"\n"));

            /* Create a session for operations */
            /*---------------------------------*/
            TST_Status = SST_SessionCreate(&TST_SessionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionCreate",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Create a new AES shared key authenticator */
            /*-------------------------------------------*/
            TST_Status = SST_AuthenticatorAESSharedSecretCreate(TST_TransactionId,
                                                               TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_AesKey,
                                                               SST_KEY_TYPE_AES_128_BIT,
                                                               &TST_AuthHandle,
                                                               TST_Workspace,
                                                               TST_WorkspaceSizeInBytes);

            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorAESSharedSecretCreate",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* re-open a transaction for operations */
            /*--------------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the authentication */
            /*-------------------------*/
            /* Get the challenge*/
            TST_Status = SST_AuthenticatorChallengeGet(TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_ChallengeBuff,
                                                      TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_ChallengeBuffSize,  
                                                      &TST_ChallengeId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorChallengeGet",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Encrypt the challenge using the AES key */
            TST_Status = CRYS_AES(DX_NULL,
                                 TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_AesKey,
                                 CRYS_AES_Key128BitSize,
                                 CRYS_AES_Encrypt,
                                 CRYS_AES_ECB_mode,
                                 TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_ChallengeBuff,
                                 TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_ChallengeBuffSize,
                                 TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_EncChallengeBuff);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the authenticator */
            TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                               TST_AuthHandle,
                                               TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_EncChallengeBuff,
                                               TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_EncChallengeBuffSize,
                                               TST_ChallengeId,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);               
                                               
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* re-open a transaction for operations */
            /*--------------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Insert a new Data object */
            /*--------------------------*/
            TST_Status = SST_DataInsert(TST_TransactionId,
                                       TST_SessionId, 
                                       TST_AuthHandle,       
                                       TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Data,
                                       TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_DataSize,
                                       DX_TRUE,
                                       TST_SimpleData,                
                                       &TST_DataHandle,
                                       TST_Workspace,
                                       TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataInsert",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);



            /* Close the authentication */
            /*--------------------------*/
            TST_Status = SST_AuthenticationClose(TST_SessionId,TST_AuthHandle);
            
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Re-open the authentication */
            /*----------------------------*/
            /* Get the challenge*/
            TST_Status = SST_AuthenticatorChallengeGet(TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_NewChallengeBuff,
                                                      TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_NewChallengeBuffSize,  
                                                      &TST_ChallengeId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorChallengeGet",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Encrypt the challenge using the AES key */
            TST_Status = CRYS_AES(DX_NULL,
                                 TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_AesKey,
                                 CRYS_AES_Key128BitSize,
                                 CRYS_AES_Encrypt,
                                 CRYS_AES_ECB_mode,
                                 TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_NewChallengeBuff,
                                 TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_NewChallengeBuffSize,
                                 TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_EncChallengeBuff);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the authenticator */
            TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                               TST_AuthHandle,
                                               TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_EncChallengeBuff,
                                               TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_EncChallengeBuffSize,
                                               TST_ChallengeId,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);                                                            
                                              

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* re-open a transaction for operations */
            /*--------------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the data */
            /*-----------------*/
             TST_Status = SST_DataDelete(TST_TransactionId,
                                        TST_SessionId,  
                                        TST_DataHandle,
                                        TST_Workspace,
                                        TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataDelete",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the authentication */
            /*--------------------------*/
            TST_Status = SST_AuthenticationClose(TST_SessionId,TST_AuthHandle);
            
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the authenticator */
            /*--------------------------*/
            TST_Status = SST_AuthenticatorDelete(TST_TransactionId,
                                                TST_AuthHandle);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorDelete",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the session */
            /*--------------------*/
            TST_Status = SST_SessionDelete(TST_SessionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionDelete",
                          TST_ATP_AesKeyAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


        }/*End of TST_index "for loop" */
    }/*End of TST_HandleType "for loop" */
    return TST_Status;

  EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}/* End of SST_TST_ATP_AesKeyAsymAuthenticator */
/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*****************************************************************************
 * Function Name:                                                           
 *  SST_TST_ATP_ECCAsymAuthenticator         
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  DxError_t - Function error return                                        
 *
 * Description:                                                                 
 * Test the creation, deletion, modification, opening, closing and usage of an ECC asymmetric key authenticator object.
 * And demonstration on the use of transaction with authenticator objects.
 *  
 * Algorithm:       
 *    1.  Build a Private Key using CRYS for CRYS usage by calling: CRYS_RSA_Build_PrivKey (externally generated key).
 *    2.  Create a new session by calling: SST_SessionCreate.
 *    3.  Open a transaction by calling: SST_TransactionStart.
 *    4.  Create a new asymmetric key authenticator by calling: SST_AuthenticatorPubKeyECCCreate.
 *    5.  Close the transaction by calling: SST_TransactionClose.
 *    6.  Open a transaction by calling: SST_TransactionStart.
 *    7.  Open the new authenticator by calling: 
 *        i.    SST_AuthenticatorChallengeGet 
 *        ii.  CRYS_ECDSA_Sign
 *        iii.  SST_AuthenticationOpen.
 *    8.  Close the transaction by calling: SST_TransactionClose.
 *    9.  Open a transaction by calling: SST_TransactionStart.
 *    10.  Insert a new data object by calling: SST_InsertData.
 *    11.  Close the authenticator by calling: SST_AuthenticationClose.
 *    12.  Open the authenticator by calling: 
 *        i.      SST_AuthenticatorChallengeGet 
 *        ii.  CRYS_ECDSA_Sign
 *        iii.  SST_AuthenticationOpen.
 *    13.  Close the transaction by calling: SST_TransactionClose.
 *    14.  Open a transaction by calling: SST_TransactionStart.
 *    15.  Delete the data by calling: SST_DeleteData.
 *    16.  Delete the authenticator by calling: SST_AuthenticatorDelete.
 *    17.  Close the transaction by calling: SST_TransactionClose.
 *    18.  Delete the session by calling: SST_SessionDelete.                           
 *
 *******************************************************************************/    
DxError_t SST_TST_ATP_ECCAsymAuthenticator ( void )
{
    /********* TST Variables Definitions ************/
    int                         TST_index,TST_HandleType;
    char                        TST_MasterTestName[TST_FUNC_STRING_LEN] = "SST_TST_ATP_ECCAsymAuthenticator";
    SSTSessionId_t              TST_SessionId; 
    SSTTxnId_t                  TST_TransactionId;
    SSTHandle_t                 TST_AuthHandle,TST_DataHandle;
    SSTChallengeId_t            TST_ChallengeId;
    SSTDataType_t               TST_SimpleData = TST_DATA_OBJECT;
    static CRYS_ECPKI_UserPrivKey_t       TST_CRYS_ECPKI_UserPrivKey;
    static CRYS_ECDSA_SignUserContext_t   TST_CRYS_ECDSA_SignUserContext;
    DxError_t                   TST_Status;
    /********* TST Variables initialization *********/

    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************************** \n"));
    ATP_LOG_TST_PRINT((SST_TST_MSG," ** SST_TST_ATP_ECCAsymAuthenticator Tests ********************* \n"));     
    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************************** \n \n"));

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    TST_Status = TESTPASS;
    for (TST_HandleType = TST_INTERNAL_HANDLE ; TST_HandleType < TST_HANDLES_TYPES ; TST_HandleType++)
    {
        for (TST_index =0 ; TST_index < TST_NumberOf_RsaAuthObjectTests ; TST_index++) /*To run on all test vectors*/
        {
            if (TST_HandleType == TST_INTERNAL_HANDLE)
            {   
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using internal handles          \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                TST_AuthHandle.objId = SST_HANDLE_RANDOM_GENERATE; 
                TST_DataHandle.objId = SST_HANDLE_RANDOM_GENERATE;
            }
            else
            {
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using external handles          \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                TST_AuthHandle.objId = TST_TEST_DEFINED_HENDLE;
                TST_DataHandle.objId = TST_TEST_DEFINED_HENDLE;
            }

            ATP_LOG_TST_PRINT((SST_TST_MSG,"\n",TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,"\n"));

            /* Build the Private Key for CRYS usage */
            /*--------------------------------------*/
            TST_Status = CRYS_ECPKI_BuildPrivKey(CRYS_ECPKI_DomainID_secp256r1,   
                                                TST_ATP_ECCAuthObjectTestData[TST_index].TST_ExternalPrivKey,
                                                TST_ATP_ECCAuthObjectTestData[TST_index].TST_ExternalPrivKeySize,
                                                &TST_CRYS_ECPKI_UserPrivKey);

            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_ECPKI_BuildPrivKey",
                                                      TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                                                      TST_MasterTestName);
  

            /* Create a session for operations */
            /*---------------------------------*/
            TST_Status = SST_SessionCreate(&TST_SessionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionCreate",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Create a new ECC authenticator */
            /*--------------------------------*/
            TST_Status = SST_AuthenticatorPubKeyECCCreate(TST_TransactionId,
                                                         TST_ATP_ECCAuthObjectTestData[TST_index].TST_ExternalPublKey,
                                                         TST_ATP_ECCAuthObjectTestData[TST_index].TST_ExternalPublKeySize,
                                                         &TST_AuthHandle,
                                                         TST_Workspace,
                                                         TST_WorkspaceSizeInBytes);

            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorPubKeyECCCreate",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* re-open a transaction for operations */
            /*--------------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);
  

            /* Open the authentication */
            /*-------------------------*/
            /* Get the challenge*/
            TST_Status = SST_AuthenticatorChallengeGet(TST_ATP_ECCAuthObjectTestData[TST_index].TST_ChallengeBuff,
                                                      TST_ATP_ECCAuthObjectTestData[TST_index].TST_ChallengeBuffSize,  
                                                      &TST_ChallengeId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorChallengeGet",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            TST_Status = CRYS_ECDSA_Sign(&TST_CRYS_ECDSA_SignUserContext,
                                        &TST_CRYS_ECPKI_UserPrivKey,
                                        CRYS_ECPKI_HASH_SHA256_mode,
                                        TST_ATP_ECCAuthObjectTestData[TST_index].TST_ChallengeBuff,      
                                        TST_ATP_ECCAuthObjectTestData[TST_index].TST_ChallengeBuffSize,
                                        TST_ATP_ECCAuthObjectTestData[TST_index].TST_SignedChallengeBuff,
                                        &TST_ATP_ECCAuthObjectTestData[TST_index].TST_SignedChallengeBuffSize);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_ECDSA_Sign",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the authenticator */
            TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                               TST_AuthHandle,
                                               TST_ATP_ECCAuthObjectTestData[TST_index].TST_SignedChallengeBuff,
                                               TST_ATP_ECCAuthObjectTestData[TST_index].TST_SignedChallengeBuffSize,
                                               TST_ChallengeId,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);               
                                               

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* re-open a transaction for operations */
            /*--------------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Insert a new Data object */
            /*--------------------------*/
            TST_Status = SST_DataInsert(TST_TransactionId,
                                       TST_SessionId, 
                                       TST_AuthHandle,       
                                       TST_ATP_ECCAuthObjectTestData[TST_index].TST_Data,
                                       TST_ATP_ECCAuthObjectTestData[TST_index].TST_DataSize,
                                       DX_TRUE,
                                       TST_SimpleData,                
                                       &TST_DataHandle,
                                       TST_Workspace,
                                       TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataInsert",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);



            /* Close the authentication */
            /*--------------------------*/
            TST_Status = SST_AuthenticationClose(TST_SessionId,TST_AuthHandle);
            
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Re-open the authentication */
            /*----------------------------*/
            /* Get the challenge*/
            TST_Status = SST_AuthenticatorChallengeGet(TST_ATP_ECCAuthObjectTestData[TST_index].TST_NewChallengeBuff,
                                                      TST_ATP_ECCAuthObjectTestData[TST_index].TST_NewChallengeBuffSize,  
                                                      &TST_ChallengeId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorChallengeGet",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Sign the Challenge buffer */
            TST_Status = CRYS_ECDSA_Sign(&TST_CRYS_ECDSA_SignUserContext,
                                        &TST_CRYS_ECPKI_UserPrivKey,
                                        CRYS_ECPKI_HASH_SHA256_mode,                
                                        TST_ATP_ECCAuthObjectTestData[TST_index].TST_NewChallengeBuff,      
                                        TST_ATP_ECCAuthObjectTestData[TST_index].TST_NewChallengeBuffSize,
                                        TST_ATP_ECCAuthObjectTestData[TST_index].TST_SignedChallengeBuff,
                                        &TST_ATP_ECCAuthObjectTestData[TST_index].TST_SignedChallengeBuffSize);


      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_ECDSA_Sign",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the authenticator */
            TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                               TST_AuthHandle,
                         TST_ATP_ECCAuthObjectTestData[TST_index].TST_SignedChallengeBuff,
                         TST_ATP_ECCAuthObjectTestData[TST_index].TST_SignedChallengeBuffSize,
                                               TST_ChallengeId,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);
                                               

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* re-open a transaction for operations */
            /*--------------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the data */
            /*-----------------*/
             TST_Status = SST_DataDelete(TST_TransactionId,
                                        TST_SessionId,  
                                        TST_DataHandle,
                                        TST_Workspace,
                                        TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataDelete",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the authentication */
            /*--------------------------*/
            TST_Status = SST_AuthenticationClose(TST_SessionId,TST_AuthHandle);
            
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the authenticator */
            /*--------------------------*/
            TST_Status = SST_AuthenticatorDelete(TST_TransactionId,
                                                TST_AuthHandle);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorDelete",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the session */
            /*--------------------*/
            TST_Status = SST_SessionDelete(TST_SessionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionDelete",
                          TST_ATP_ECCAuthObjectTestData[TST_index].TST_Name,
                          TST_MasterTestName);


        }/*End of TST_index "for loop" */
    }/*End of TST_HandleType "for loop" */
    return TST_Status;

  EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}/* End of SST_TST_ATP_ECCAsymAuthenticator */
/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/

#endif

